<?php 
/*
Plugin Name: Auto SEO
Plugin URI: 
Description: SEO 插件
Version: 2025.1.6
Author: 吴杰
Author URI: https://wujie.me
License: GPL
*/

// 移除字符串末尾的标点符号
function remove_trailing_punctuation($text) {
    // 使用正则表达式移除末尾的标点符号
    return preg_replace('/[.,!?，。！？…；]+$/u', '', $text);
}

// 生成meta标签的函数
function autoseo_generate_meta_tag($name, $content) {
    echo '<meta name="' . esc_attr($name) . '" content="' . esc_attr($content) . '">' . "\n";
}

// 获取关键词的函数
function get_post_keywords($post) {
    $keywords = array();
    
    // 1. 获取标签
    $tags = get_the_tags($post->ID);
    if ($tags) {
        foreach ($tags as $tag) {
            if (mb_strlen($tag->name, 'UTF-8') > 1) {
                $keywords[] = $tag->name;
            }
        }
    }
    
    // 2. 获取分类
    $categories = get_the_category($post->ID);
    foreach ($categories as $category) {
        if ($category->name != '未分类' && mb_strlen($category->name, 'UTF-8') > 1) {
            $keywords[] = $category->name;
        }
    }
    
    // 3. 从标题中提取关键词
    $title = get_the_title($post->ID);
    $title_keywords = mb_split('[\s,，]', $title);
    foreach ($title_keywords as $keyword) {
        if (mb_strlen($keyword, 'UTF-8') > 1) {
            $keywords[] = $keyword;
        }
    }
    
    // 限制关键词数量为5个
    $keywords = array_slice(array_unique($keywords), 0, 5);
    
    return implode(',', $keywords);
}

// 添加meta标签的函数
function add_meta_tags(): void
{
    if (is_single() || is_page()) {
        global $post;
        
        // 检查文章是否为私密或加密
        if ($post->post_status === 'private' || !empty($post->post_password)) {
            return;
        }
        
        // 获取文章摘要
        if (has_excerpt($post->ID)) {
            // 如果有手动填写的摘要，直接使用，并去掉空格和换行符
            $description = get_the_excerpt($post->ID);
            $description = preg_replace('/\s+/', ' ', $description);
            // 判断最后是不是省略号或...结尾，如果不是，则加上省略号
            if (!preg_match('/(\.\.\.|…|……)$/', $description)) {
                $description = remove_trailing_punctuation($description) . '...';
            }
        } else {
            // 如果没有手动填写摘要，使用自动摘要的第一句话
            $content = strip_shortcodes($post->post_content);
            $content = wp_strip_all_tags($content);
            
            // 去掉多余的空行和空格
            $content = preg_replace('/\n\s*\n/m', ' ', $content);
            $content = preg_replace('/\s+/', ' ', $content);
            
            // 首先分割成句子，使用更严格的正则表达式
            $sentences = preg_split('/(?<=[。！？\.])\s*/u', $content, -1, PREG_SPLIT_NO_EMPTY);
            
            if (!empty($sentences)) {
                
                // 获取第一句话
                $description = trim($sentences[0]);
                
                // 如果第一句话超过200个字，取前4个逗号前的内容
                if (mb_strlen($description, 'UTF-8') > 200) {
                    $parts = mb_split('，', $description);
                    $description = '';
                    for ($i = 0; $i < min(4, count($parts)); $i++) {
                        $description .= $parts[$i] . '，';
                    }
                    $description = rtrim($description, '，');
                } else {
                    // 如果有第二句话，获取第一个逗号前的内容
                    if (isset($sentences[1])) {
                        $second_sentence = trim($sentences[1]);
                        
                        $comma_pos = mb_strpos($second_sentence, '，', 0, 'utf-8');
                        if ($comma_pos !== false) {
                            $description .= mb_substr($second_sentence, 0, $comma_pos + 1, 'utf-8');
                        }
                    }
                }

                // 去掉空格和换行符
                $description = preg_replace('/\s+/', ' ', $description);
                
                // 判断最后是不是省略号或...结尾，如果不是，则加上省略号
                if (!preg_match('/(\.\.\.|…|……)$/', $description)) {
                    $description = remove_trailing_punctuation($description) . '...';
                }                
            }
        }

        // 获取关键词
        $keywords = get_post_keywords($post);
        if (empty($keywords)) {
            $keywords = get_the_title($post->ID);
        }

        // 生成meta标签
        if ($description) {
            autoseo_generate_meta_tag('description', $description);
        }
        if ($keywords) {
            autoseo_generate_meta_tag('keywords', $keywords);
        }
    } else {
        // 如果不是单篇文章或页面，使用默认的关键词和描述
        autoseo_generate_meta_tag('keywords', '慢读时光, 读书博客, 生活博客, 吴杰博客, 阅读心得, 生活记录');
        autoseo_generate_meta_tag('description', '慢读时光博客，记录平凡生活，分享读书感悟，探索生活中的美好瞬间。');
    }
}
// 将add_meta_tags函数挂载到wp_head钩子上
add_action('wp_head', 'add_meta_tags');
